import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:pull_to_refresh/pull_to_refresh.dart';
import '../../domain/models/models.dart';
import '../../domain/repository/repository.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import 'comments_state.dart';

class CommentsNotifier extends StateNotifier<CommentsState> {
  final CommentsRepository _commentsRepository;
  int _page = 0;
  int _shopPage = 0;

  CommentsNotifier(this._commentsRepository) : super(const CommentsState());

  void setActiveIndex(int index) {
    if (state.activeIndex == index) {
      return;
    }
    state = state.copyWith(activeIndex: index);
  }

  Future<void> fetchProductComments({
    BuildContext? context,
    bool? isRefresh,
  }) async {
    if (isRefresh ?? false) {
      _page = 0;
      state = state.copyWith(
        hasMoreProductComment: true,
        productComments: [],
        isLoading: true,
      );
    } else {
      state = state.copyWith(hasMoreLoading: true);
    }
    final res = await _commentsRepository.getProductComments(page: ++_page);
    res.when(
      success: (data) {
        List<CommentsData> list = List.from(state.productComments);
        list.addAll(data.data ?? []);
        state = state.copyWith(hasMoreLoading: false);

        state = state.copyWith(isLoading: false, productComments: list);
        if ((data.data?.length ?? 0) < 12) {
          state = state.copyWith(hasMoreProductComment: false);
        }
      },
      failure: (failure, status) {
        state = state.copyWith(isLoading: false);
        if (context != null) {
          AppHelpers.showSnackBar(context, failure);
        }
      },
    );
  }

  void changeState(int index) {
    state = state.copyWith(stateIndex: index);
  }

  Future<void> fetchShopComments({
    BuildContext? context,
    bool? isRefresh,
    RefreshController? controller,
  }) async {
    if (isRefresh ?? false) {
      _shopPage = 0;
      state = state.copyWith(
        hasMoreShopComment: true,
        shopComments: [],
        isLoading: true,
      );
    } else {
      state = state.copyWith(hasMoreLoading: true);
    }
    final res = await _commentsRepository.getShopComments(page: ++_shopPage);
    res.when(
      success: (data) {
        List<CommentsData> list = List.from(state.shopComments);
        list.addAll(data.data ?? []);
        state = state.copyWith(hasMoreLoading: false);

        state = state.copyWith(isLoading: false, shopComments: list);
        if ((data.data?.length ?? 0) < 12) {
          state = state.copyWith(hasMoreShopComment: false);
        }
      },
      failure: (failure, status) {
        state = state.copyWith(isLoading: false);
        if (context != null) {
          AppHelpers.showSnackBar(context, failure);
        }
      },
    );
  }

  void changeSelectProductComment(int index) {
    if (index == state.selectedProductComment) {
      state = state.copyWith(selectedProductComment: -1);
    } else {
      state = state.copyWith(selectedProductComment: index);
    }
  }

  void changeSelectShopComment(int index) {
    if (index == state.selectedShopComment) {
      state = state.copyWith(selectedShopComment: -1);
    } else {
      state = state.copyWith(selectedShopComment: index);
    }
  }

  void addSelectProduct({int? id, required int orderLength}) {
    List<int> list = List.from(state.selectedProductIds);
    if (list.contains(id)) {
      list.remove(id);
    } else {
      list.add(id ?? 0);
    }
    state = state.copyWith(selectedProductIds: list);

    if (list.length == orderLength) {
      state = state.copyWith(isAllSelectProduct: true);
    } else {
      state = state.copyWith(isAllSelectProduct: false);
    }
  }

  void addSelectShop({int? id, required int orderLength}) {
    List<int> list = List.from(state.selectedShopIds);
    if (list.contains(id)) {
      list.remove(id);
    } else {
      list.add(id ?? 0);
    }
    state = state.copyWith(selectedShopIds: list);

    if (list.length == orderLength) {
      state = state.copyWith(isAllSelectShop: true);
    } else {
      state = state.copyWith(isAllSelectShop: false);
    }
  }

  void allSelectProduct(List<CommentsData> comments) {
    List<int> list = [];
    if (!state.isAllSelectProduct) {
      for (int i = 0; i < comments.length; i++) {
        list.add(comments[i].id ?? 0);
      }
      state = state.copyWith(isAllSelectProduct: true);
    } else {
      state = state.copyWith(isAllSelectProduct: false);
    }
    state = state.copyWith(selectedProductIds: list);
  }

  void allSelectShop(List<CommentsData> comments) {
    List<int> list = [];
    if (!state.isAllSelectShop) {
      for (int i = 0; i < comments.length; i++) {
        list.add(comments[i].id ?? 0);
      }
      state = state.copyWith(isAllSelectShop: true);
    } else {
      state = state.copyWith(isAllSelectShop: false);
    }
    state = state.copyWith(selectedShopIds: list);
  }

  Future<void> deleteComment(BuildContext context, int id) async {
    final response = await _commentsRepository.deleteComment(id: id);
    response.when(
      success: (data) {
        List<CommentsData> productList = List.from(state.productComments);
        productList.removeWhere((element) => element.id == id);
        List<CommentsData> shopList = List.from(state.shopComments);
        shopList.removeWhere((element) => element.id == id);
        state = state.copyWith(
          productComments: productList,
          shopComments: shopList,
        );
        AppHelpers.showSnackBar(
          context,
          AppHelpers.getTranslation(TrKeys.deleted),
        );
      },
      failure: (failure, status) {
        AppHelpers.showSnackBar(context, failure);
      },
    );
  }
}
